const Joi = require("joi");
const bcrypt = require("bcryptjs");
const { Op } = require("sequelize");
const { Admins, Parcel } = require("../models");

const createAdmin = async (req, res) => {
  try {
    const { username, password } = req.body;

    const schema = Joi.object({
      username: Joi.string().required(),
      password: Joi.string().required(),
    });

    const { error } = schema.validate(req.body);
    if (error) {
      return res.status(400).json({
        success: false,
        message: "Validation error",
        errors: error.details.map((detail) => detail.message),
      });
    }

    // Check if admin already exists
    const admin = await Admins.findOne({
      where: { username },
      attributes: {
        exclude: ["password"],
      },
    });
    if (admin) {
      return res.status(400).json({
        success: false,
        message: `Unique Constraint Error`,
        errors: [`Admin with username ${username} already exists.`],
      });
    }

    const newAdmin = await Admins.create({
      username,
      password,
    });

    return res.status(201).json({
      success: true,
      message: "Admin created successfully",
      data: newAdmin,
    });
  } catch (error) {
    console.error("Error creating admin:", error);
    return res.status(500).json({
      success: false,
      message: "Internal server error",
      errors: ["An error occurred while creating admin."],
    });
  }
};

const getAdmins = async (req, res) => {
  try {
    let { page } = req.query;
    page = page ? parseInt(page) : 1;
    const limit = 50;
    const startIndex = (page - 1) * limit;
    const endIndex = page * limit;

    const schema = Joi.object({
      page: Joi.number(),
    });

    const { error } = schema.validate(req.query);
    if (error) {
      return res.status(400).json({
        success: false,
        message: "Validation error",
        errors: error.details.map((detail) => detail.message),
      });
    }

    let allAdmins = await Admins.findAndCountAll({
      limit,
      offset: startIndex,
      order: [["id", "DESC"]],
      attributes: {
        exclude: ["password"],
      },
    });

    allAdmins.limit = limit;
    const totalPages = Math.ceil(allAdmins.count / limit);
    allAdmins.totalPages = totalPages > 0 ? totalPages : 1;

    if (endIndex < allAdmins.count) {
      allAdmins.next = page + 1;
    }

    if (startIndex > 0) {
      allAdmins.previous = page - 1;
    }

    return res.status(200).json({
      success: true,
      message: "Admins retrieved successfully",
      data: allAdmins,
    });
  } catch (error) {
    console.error("Error retrieving admins:", error);
    return res.status(500).json({
      success: false,
      message: "Internal server error",
      errors: ["An error occurred while retrieving admins."],
    });
  }
};

const getAdmin = async (req, res) => {
  try {
    const { adminId } = req.admin;

    let admin = await Admins.findOne({
      where: { id: adminId },
      attributes: {
        exclude: ["password"],
      },
    });

    if (admin) {
      return res.status(200).json({
        success: true,
        message: `Admin ${adminId} retrieved successfully`,
        data: admin,
      });
    }

    return res.status(404).json({
      success: false,
      message: `Not found`,
      errors: [`Admin ${adminId} was not found`],
    });
  } catch (error) {
    console.error("Error retrieving admin:", error);
    return res.status(500).json({
      success: false,
      message: "Internal server error",
      errors: ["An error occurred while retrieving admin."],
    });
  }
};

const updateAdminPassword = async (req, res) => {
  try {
    const { adminId } = req.admin;

    const { password, newPassword } = req.body;

    const schema = Joi.object({
      password: Joi.string().required(),
      newPassword: Joi.string().required(),
    });

    const { error } = schema.validate(req.body);
    if (error) {
      return res.status(400).json({
        success: false,
        message: "Validation error",
        errors: error.details.map((detail) => detail.message),
      });
    }

    const admin = await Admins.findOne({
      where: { id: adminId },
    });

    if (admin) {
      const validatePassword = await bcrypt.compare(password, admin.password);
      if (validatePassword) {
        await Admins.update(
          {
            password: newPassword,
          },
          {
            where: { id: adminId },
          }
        );
        return res.status(200).json({
          success: true,
          message: `Admin ${adminId} password updated successfully`,
          data: [1],
        });
      }
      return res.status(400).json({
        success: false,
        message: "Incorrect password",
        errors: [`Password provided is incorrect`],
      });
    }

    return res.status(404).json({
      success: false,
      message: "Not found",
      errors: ["Admin was not found"],
    });
  } catch (error) {
    console.error("Error updating admin password:", error);
    return res.status(500).json({
      success: false,
      message: "Internal server error",
      errors: ["An error occurred while updating admin password."],
    });
  }
};

const deleteAdmin = async (req, res) => {
  try {
    const { adminId } = req.params;

    const schema = Joi.object({
      adminId: Joi.number().required(),
    });

    const { error } = schema.validate(req.params);
    if (error) {
      return res.status(400).json({
        success: false,
        message: "Validation error",
        errors: error.details.map((detail) => detail.message),
      });
    }

    let admin = await Admins.findOne({
      where: { id: adminId },
    });

    if (admin) {
      let deleteAdmin = await Users.destroy({
        where: { id: adminId },
      });
      return res.status(200).json({
        success: true,
        message: `Admin ${adminId} deleted successfully`,
        data: deleteAdmin,
      });
    }

    return res.status(404).json({
      success: false,
      message: `Not found`,
      errors: ["Admin ${adminId} was not found"],
    });
  } catch (error) {
    console.error("Error deleting admin:", error);
    return res.status(500).json({
      success: false,
      message: "Internal server error",
      errors: ["An error occurred while deleting admin."],
    });
  }
};

const siteStat = async (req, res) => {
  try {
    let totalParcel = await Parcel.count();
    let deliveredParcel = await Parcel.count({
      where: { parcelStatus: "Delivered" },
    });
    let inTransitParcel = await Parcel.count({
      where: {
        parcelStatus: {
          [Op.or]: ["In-Transit", "Onhold"],
        },
      },
    });
    let cancelledParcel = await Parcel.count({
      where: { parcelStatus: "Cancelled" },
    });

    return res.status(200).json({
      success: true,
      message: "Admins count retrieved successfully",
      data: {
        totalParcel,
        deliveredParcel,
        inTransitParcel,
        cancelledParcel,
      },
    });
  } catch (error) {
    console.error("Error counting admins:", error);
    return res.status(500).json({
      success: false,
      message: "Internal server error",
      errors: ["An error occurred while counting admins."],
    });
  }
};

module.exports = {
  createAdmin,
  getAdmins,
  getAdmin,
  updateAdminPassword,
  deleteAdmin,
  siteStat,
};
