const Joi = require('joi');
const { Op } = require('sequelize');
const { Parcel, ParcelRoutes } = require('../models');
const generateDoc = require('../utils/generateDoc');


const createParcel = async (req, res) => {
    try {
        const {
            trackingCode,
            parcelShipped,
            senderName,
            senderAddress,
            senderPhoneNumber,
            senderEmailAddress,
            receiverName,
            receiverAddress,
            receiverPhoneNumber,
            receiverEmailAddress,
            parcelPaymentMode,
            parcelWeight,
            parcelOrigin,
            parcelFinalDestination,
            parcelShippingType,
            parcelShippingMode,
            parcelShippingDate,
            parcelExpectedDeliveryDate,
            parcelStatus,
            paymentAmount,
            isPaid
        } = req.body;

        const schema = Joi.object({
            trackingCode: Joi.string().required(),
            parcelShipped: Joi.string().required(),
            senderName: Joi.string().required(),
            senderAddress: Joi.string().required(),
            senderPhoneNumber: Joi.string().required(),
            senderEmailAddress: Joi.string().required(),
            receiverName: Joi.string().required(),
            receiverAddress: Joi.string().required(),
            receiverPhoneNumber: Joi.string().required(),
            receiverEmailAddress: Joi.string().required(),
            parcelPaymentMode: Joi.string().required(),
            parcelWeight: Joi.string().required(),
            parcelOrigin: Joi.string().required(),
            parcelFinalDestination: Joi.string().required(),
            parcelShippingType: Joi.string().allow(''),
            parcelShippingMode: Joi.string().required(),
            parcelShippingDate: Joi.date().required(),
            parcelExpectedDeliveryDate: Joi.date().required(),
            parcelStatus: Joi.string().required(),
            paymentAmount: Joi.number().required(),
            isPaid: Joi.boolean().required()
        });

        const { error } = schema.validate(req.body);
        if(error){
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Validation error',
                errors: error.details.map(detail => detail.message),
            });
        }

        const newParcel = await Parcel.create({
            trackingCode,
            parcelShipped,
            senderName,
            senderAddress,
            senderPhoneNumber,
            senderEmailAddress,
            receiverName,
            receiverAddress,
            receiverPhoneNumber,
            receiverEmailAddress,
            parcelPaymentMode,
            parcelWeight,
            parcelOrigin,
            parcelFinalDestination,
            parcelShippingType,
            parcelShippingMode,
            parcelShippingDate,
            parcelExpectedDeliveryDate,
            parcelStatus,
            paymentAmount,
            isPaid
        });

        return res.status(201).json({
            status: 201,
            success: true,
            message: 'Parcel created successfully',
            data: newParcel
        });
    } catch (error) {
        console.error('Error creating parcel:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Internal server error',
            errors: ['An error occurred while creating parcel.'],
        });
    }
};




const addParcelRoute = async (req, res) => {
    try {
        const {
            from,
            to,
            status,
            location,
            pointerValue,
            parcelId
        } = req.body;

        const schema = Joi.object({
            from: Joi.string().required(),
            to: Joi.string().required(),
            status: Joi.string().required(),
            location: Joi.string().required(),
            pointerValue: Joi.number().required(),
            parcelId: Joi.string().uuid().required()
        });

        const { error } = schema.validate(req.body);
        if(error){
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Validation error',
                errors: error.details.map(detail => detail.message),
            });
        }

        const parcel = await Parcel.findOne({
            where: { parcelId }
        });

        if(!parcel){
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Not found',
                errors: [`Parcel ${parcelId} was not found`]
            });
        }

        const newParcelRoute = await ParcelRoutes.create({
            from,
            to,
            status,
            location,
            pointerValue,
            parcelId: parcel.id
        });

        return res.status(201).json({
            status: 201,
            success: true,
            message: 'Parcel Route added created successfully',
            data: newParcelRoute
        });
        
    } catch (error) {
        console.error('Error creating parcel:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Internal server error',
            errors: ['An error occurred while creating parcel.'],
        });
    }
};




const getParcel = async (req, res) => {
    try {
        const { trackingCode } = req.params;

        const schema = Joi.object({
            trackingCode: Joi.string().required()
        });

        const { error } = schema.validate(req.params);
        if(error){
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Validation error',
                errors: error.details.map(detail => detail.message),
            });
        }

        const parcel = await Parcel.findOne({
            where: { trackingCode },
            include: ParcelRoutes
        });

        if(!parcel){
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Not found',
                errors: [`Parcel ${trackingCode} was not found`]
            });
        }

        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Parcel retrieved successfully',
            data: parcel
        });

    } catch (error) {
        console.error('Error retrieving parcel:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Internal server error',
            errors: ['An error occurred while retrieving parcel.'],
        });
    }
};




const getParcels = async (req, res) => {
    try {
        let { page, limit, trackingCode } = req.query;
        page = page ? parseInt(page) : 1;
        limit = limit ?  parseInt(limit) : 50;
        const startIndex = (page - 1) * limit;
        const endIndex = page * limit;

        const schema = Joi.object({
            page: Joi.number(),
            limit: Joi.number(),
            trackingCode: Joi.string().allow('')
        });

        const { error } = schema.validate({ page, limit, trackingCode });
        if(error){
            return res.status(400).json({
                success: false,
                message: 'Validation error',
                errors: error.details.map(detail => detail.message),
            });
        }

        let whereClause = {};
        if (trackingCode) {
            whereClause.trackingCode = {
                [Op.substring]: trackingCode
            };
        }

        let parcels = await Parcel.findAndCountAll({
            distinct: true,
            limit,
            offset: startIndex,
            order: [['id', 'DESC']],
            where: {
                ...whereClause
            },
            include: ParcelRoutes
        });

        parcels.limit = limit;
        const totalPages = Math.ceil(parcels.count/limit);
        parcels.totalPages = (totalPages > 0) ? totalPages : 1;

        if(endIndex < parcels.count){
            parcels.next = page + 1;
        }

        if(startIndex > 0){
            parcels.previous = page - 1;
        }

        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Parcels retrieved successfully',
            data: parcels
        });

    } catch (error) {
        console.error('Error retrieving parcels:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Internal server error',
            errors: ['An error occurred while retrieving parcels.'],
        });
    }
};




const updateParcel = async (req, res) => {
    try {
        const { trackingCode } = req.params;

        const {
            parcelShipped,
            senderName,
            senderAddress,
            senderPhoneNumber,
            senderEmailAddress,
            receiverName,
            receiverAddress,
            receiverPhoneNumber,
            receiverEmailAddress,
            parcelPaymentMode,
            parcelWeight,
            parcelOrigin,
            parcelFinalDestination,
            parcelShippingType,
            parcelShippingMode,
            parcelShippingDate,
            parcelExpectedDeliveryDate,
            parcelStatus,
            paymentAmount,
            isPaid
        } = req.body;

        const schema = Joi.object({
            trackingCode: Joi.string(),
            parcelShipped: Joi.string(),
            senderName: Joi.string(),
            senderAddress: Joi.string(),
            senderPhoneNumber: Joi.string(),
            senderEmailAddress: Joi.string(),
            receiverName: Joi.string(),
            receiverAddress: Joi.string(),
            receiverPhoneNumber: Joi.string(),
            receiverEmailAddress: Joi.string(),
            parcelPaymentMode: Joi.string(),
            parcelWeight: Joi.string(),
            parcelOrigin: Joi.string().required(),
            parcelFinalDestination: Joi.string().required(),
            parcelShippingType: Joi.string(),
            parcelShippingMode: Joi.string(),
            parcelShippingDate: Joi.date(),
            parcelExpectedDeliveryDate: Joi.date(),
            parcelStatus: Joi.string(),
            paymentAmount: Joi.number(),
            isPaid: Joi.boolean()
        });

        const { error } = schema.validate({ ...req.body, ...req.params });
        if(error){
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Validation error',
                errors: error.details.map(detail => detail.message),
            });
        }

        const parcel = await Parcel.findOne({
            where: { trackingCode }
        });

        if(!parcel){
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Not found',
                errors: [`Parcel ${trackingCode} was not found`]
            });
        }

        if(parcelShipped){
            parcel.parcelShipped = parcelShipped;
        }

        if(senderName){
            parcel.senderName = senderName;
        }

        if(senderAddress){
            parcel.senderAddress = senderAddress;
        }

        if(senderPhoneNumber){
            parcel.senderPhoneNumber = senderPhoneNumber;
        }

        if(senderEmailAddress){
            parcel.senderEmailAddress = senderEmailAddress;
        }

        if(receiverName){
            parcel.receiverName = receiverName;
        }

        if(receiverAddress){
            parcel.receiverAddress = receiverAddress;
        }

        if(receiverPhoneNumber){
            parcel.receiverPhoneNumber = receiverPhoneNumber;
        }

        if(receiverEmailAddress){
            parcel.receiverEmailAddress = receiverEmailAddress;
        }

        if(parcelPaymentMode){
            parcel.parcelPaymentMode = parcelPaymentMode;
        }
   
        if(parcelWeight){
            parcel.parcelWeight = parcelWeight;
        }

        if(parcelOrigin){
            parcel.parcelOrigin = parcelOrigin;
        }

        if(parcelFinalDestination){
            parcel.parcelFinalDestination = parcelFinalDestination;
        }

        if(parcelShippingType){
            parcel.parcelShippingType = parcelShippingType;
        }

        if(parcelShippingMode){
            parcel.parcelShippingMode = parcelShippingMode;
        }

        if(parcelShippingDate){
            parcel.parcelShippingDate = parcelShippingDate;
        }

        if(parcelExpectedDeliveryDate){
            parcel.parcelExpectedDeliveryDate = parcelExpectedDeliveryDate;
        }

        if(parcelStatus){
            parcel.parcelStatus = parcelStatus;
        }

        if(paymentAmount){
            parcel.paymentAmount = paymentAmount;
        }

        if(isPaid){
            parcel.isPaid = isPaid;
        }

        await parcel.save();

        return res.status(200).json({
            status: 200,
            success: true,
            message: `Parcel ${trackingCode} updated successfully`,
            data: parcel
        });

    } catch (error) {
        console.error('Error updating parcel:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Internal server error',
            errors: ['An error occurred while updating parcel.'],
        });
    }
};



const updateParcelRoute = async (req, res) => {
    try {
        const { parcelRouteId } = req.params;

        const {
            from,
            to,
            status,
            location,
            pointerValue
        } = req.body;

        const schema = Joi.object({
            parcelRouteId: Joi.string().uuid().required(),
            from: Joi.string().required(),
            to: Joi.string().required(),
            status: Joi.string().required(),
            location: Joi.string().required(),
            pointerValue: Joi.number().required()
        });

        const { error } = schema.validate({ ...req.body, ...req.params });
        if(error){
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Validation error',
                errors: error.details.map(detail => detail.message),
            });
        }

        const parcelRoute = await ParcelRoutes.findOne({
            where: { parcelRouteId }
        });

        if(!parcelRoute){
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Not found',
                errors: [`Parcel Route ${parcelRouteId} was not found`]
            });
        }

        if(from){
            parcelRoute.from = from;
        }

        if(to){
            parcelRoute.to = to;
        }

        if(status){
            parcelRoute.status = status;
        }

        if(location){
            parcelRoute.location = location;
        }

        if(pointerValue){
            parcelRoute.pointerValue = pointerValue;
        }

        await parcelRoute.save();

        return res.status(200).json({
            status: 200,
            success: true,
            message: `Parcel Route ${parcelRouteId} updated successfully`,
            data: parcelRoute
        });

    } catch (error) {
        console.error('Error updating parcel route:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Internal server error',
            errors: ['An error occurred while updating parcel route.'],
        });
    }
};



const deleteParcel = async (req, res) => {
    try {
        const { trackingCode } = req.params;

        const schema = Joi.object({
            trackingCode: Joi.string().required()
        });

        const { error } = schema.validate(req.params);
        if(error){
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Validation error',
                errors: error.details.map(detail => detail.message),
            });
        }

        const parcel = await Parcel.findOne({
            where: { trackingCode }
        });

        if(!parcel){
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Not found',
                errors: [`Parcel ${trackingCode} was not found`]
            });
        }

        await Parcel.destroy({
            where: { trackingCode }
        });

        return res.status(200).json({
            status: 200,
            success: true,
            message: `Parcel ${trackingCode} deleted successfully`,
            data: null
        });

    } catch (error) {
        console.error('Error deleting parcel:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Internal server error',
            errors: ['An error occurred while deleting parcel.'],
        });
    }
};



const deleteParcelRoute = async (req, res) => {
    try {
        const { parcelRouteId } = req.params;

        const schema = Joi.object({
            parcelRouteId: Joi.string().uuid().required()
        });

        const { error } = schema.validate(req.params);
        if(error){
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Validation error',
                errors: error.details.map(detail => detail.message),
            });
        }

        const parcelRoute = await ParcelRoutes.findOne({
            where: { parcelRouteId }
        });

        if(!parcelRoute){
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Not found',
                errors: [`Parcel Route ${parcelRouteId} was not found`]
            });
        }

        await ParcelRoutes.destroy({
            where: { parcelRouteId }
        });

        return res.status(200).json({
            status: 200,
            success: true,
            message: `Parcel Route ${parcelRouteId} deleted successfully`,
            data: null
        });

    } catch (error) {
        console.error('Error deleting parcel route:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Internal server error',
            errors: ['An error occurred while deleting parcel route.'],
        });
    }
}



const generateInvoice = async (req, res) => {
    try {
        const { parcelId } = req.params;

        const schema = Joi.object({
            parcelId: Joi.string().uuid().required()
        });

        const { error } = schema.validate(req.params);
        if(error){
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Validation error',
                errors: error.details.map(detail => detail.message),
            });
        }

        const generateInvoice = await generateDoc('invoice-template.html', parcelId);

        return res.status(200).json({
            status: 200,
            success: true,
            message: `Invoice generated successfully`,
            data: generateInvoice
        });
        
    } catch (error) {
        console.error('Error generating invoice:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Internal server error',
            errors: ['An error occurred while generating invoice.'],
        });
    }
}



const generateReceipt = async (req, res) => {
    try {
        const { parcelId } = req.params;

        const schema = Joi.object({
            parcelId: Joi.string().uuid().required()
        });

        const { error } = schema.validate(req.params);
        if(error){
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Validation error',
                errors: error.details.map(detail => detail.message),
            });
        }

        const generateReceipt = await generateDoc('receipt-template.html', parcelId);

        return res.status(200).json({
            status: 200,
            success: true,
            message: `Receipt generated successfully`,
            data: generateReceipt
        });
        
    } catch (error) {
        console.error('Error generating receipt:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Internal server error',
            errors: ['An error occurred while generating receipt.'],
        });
    }
}



module.exports = {
    createParcel,
    addParcelRoute,
    getParcel,
    getParcels,
    updateParcel,
    updateParcelRoute,
    deleteParcel,
    deleteParcelRoute,
    generateInvoice,
    generateReceipt
};
